import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import pandas as pd
import os
import threading
import sys

class ExcelMergeApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Excel Merge Tool")
        self.root.geometry("600x500")
        
        # Variables
        self.source_folder = tk.StringVar()
        self.output_file = tk.StringVar()
        self.file_list = []
        
        # UI Setup
        self.create_widgets()
        
        # Style
        style = ttk.Style()
        style.configure("TButton", padding=6)
        style.configure("TLabel", padding=5)

    def create_widgets(self):
        # Main Container
        main_frame = ttk.Frame(self.root, padding="20")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # --- Section 1: Input Source ---
        lbl_instruction = ttk.Label(main_frame, text="Step 1: Select folder containing Excel files", font=("Segoe UI", 10, "bold"))
        lbl_instruction.pack(anchor="w", pady=(0, 5))
        
        input_frame = ttk.Frame(main_frame)
        input_frame.pack(fill=tk.X, pady=(0, 15))
        
        btn_browse_folder = ttk.Button(input_frame, text="Browse Folder...", command=self.browse_folder)
        btn_browse_folder.pack(side=tk.LEFT)
        
        self.lbl_source = ttk.Label(input_frame, text="No folder selected", foreground="gray")
        self.lbl_source.pack(side=tk.LEFT, padx=10)
        
        # --- Section 2: File List Preview ---
        lbl_files = ttk.Label(main_frame, text="Found Files:", font=("Segoe UI", 9))
        lbl_files.pack(anchor="w")
        
        list_frame = ttk.Frame(main_frame)
        list_frame.pack(fill=tk.BOTH, expand=True, pady=5)
        
        scrollbar = ttk.Scrollbar(list_frame)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.listbox = tk.Listbox(list_frame, yscrollcommand=scrollbar.set, height=10, selectmode=tk.EXTENDED)
        self.listbox.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        scrollbar.config(command=self.listbox.yview)
        
        # --- Section 3: Output ---
        lbl_output = ttk.Label(main_frame, text="Step 2: Select output file", font=("Segoe UI", 10, "bold"))
        lbl_output.pack(anchor="w", pady=(15, 5))
        
        output_frame = ttk.Frame(main_frame)
        output_frame.pack(fill=tk.X, pady=(0, 15))
        
        btn_save_as = ttk.Button(output_frame, text="Select Destination...", command=self.save_as)
        btn_save_as.pack(side=tk.LEFT)
        
        self.lbl_output_path = ttk.Label(output_frame, text="No output file selected", foreground="gray")
        self.lbl_output_path.pack(side=tk.LEFT, padx=10)
        
        # --- Section 4: Action ---
        self.status_var = tk.StringVar(value="Ready")
        lbl_status = ttk.Label(main_frame, textvariable=self.status_var, relief=tk.SUNKEN, anchor="w")
        lbl_status.pack(side=tk.BOTTOM, fill=tk.X)

        self.btn_merge = ttk.Button(main_frame, text="MERGE FILES NOW", command=self.start_merge_thread, state=tk.DISABLED)
        self.btn_merge.pack(side=tk.BOTTOM, pady=20, fill=tk.X)
        
    def browse_folder(self):
        folder = filedialog.askdirectory()
        if folder:
            self.source_folder.set(folder)
            self.lbl_source.config(text=folder, foreground="black")
            self.scan_files(folder)
            self.check_ready()

    def scan_files(self, folder):
        self.listbox.delete(0, tk.END)
        self.file_list = []
        try:
            files = [f for f in os.listdir(folder) if f.endswith(('.xlsx', '.xls')) and not f.startswith('~$')]
            files.sort()
            for f in files:
                self.listbox.insert(tk.END, f)
                self.file_list.append(os.path.join(folder, f))
            
            self.status_var.set(f"Found {len(self.file_list)} Excel files.")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to scan folder: {e}")

    def save_as(self):
        f = filedialog.asksaveasfilename(defaultextension=".xlsx", filetypes=[("Excel Files", "*.xlsx")])
        if f:
            self.output_file.set(f)
            self.lbl_output_path.config(text=os.path.basename(f), foreground="black")
            self.check_ready()

    def check_ready(self):
        if self.file_list and self.output_file.get():
            self.btn_merge.config(state=tk.NORMAL)
        else:
            self.btn_merge.config(state=tk.DISABLED)

    def start_merge_thread(self):
        self.btn_merge.config(state=tk.DISABLED)
        threading.Thread(target=self.run_merge, daemon=True).start()

    def run_merge(self):
        try:
            all_dfs = []
            files = self.file_list
            total = len(files)
            
            for index, file_path in enumerate(files):
                self.status_var.set(f"Processing ({index+1}/{total}): {os.path.basename(file_path)}...")
                try:
                    df = pd.read_excel(file_path)
                    # Optional: Add a column for source file name
                    df['Source_File'] = os.path.basename(file_path)
                    all_dfs.append(df)
                except Exception as e:
                    print(f"Error reading {file_path}: {e}")
            
            self.status_var.set("Merging dataframes...")
            if not all_dfs:
                messagebox.showwarning("Warning", "No valid data found in files.")
                self.reset_ui()
                return

            master_df = pd.concat(all_dfs, ignore_index=True)
            
            self.status_var.set("Saving master file...")
            master_df.to_excel(self.output_file.get(), index=False)
            
            self.status_var.set("Done!")
            messagebox.showinfo("Success", f"Successfully merged {total} files!\n\nSaved to:\n{self.output_file.get()}")
            
        except Exception as e:
            messagebox.showerror("Error", f"An error occurred during merge:\n{str(e)}")
            self.status_var.set("Error occurred.")
        finally:
            self.reset_ui()

    def reset_ui(self):
        self.root.after(0, lambda: self.btn_merge.config(state=tk.NORMAL))

if __name__ == "__main__":
    root = tk.Tk()
    app = ExcelMergeApp(root)
    root.mainloop()
